﻿using System;
using System.Runtime.InteropServices;
using System.Globalization;


namespace VPM_API
{
    /// <summary>
    /// Wrapper class for talking to the alphatronics port monitor.
    /// The fields except for "size" are all C++ char[]--of the same size.
    /// If you have a need for something larger than MAX_PATH, change the
    /// const "StringSize"
    /// </summary>
    [StructLayout(LayoutKind.Sequential, CharSet = CharSet.Unicode)]
    public unsafe class AVPMonitor
    {
        protected UInt32 Size_;              // Required; total bytes in this structure
        protected IntPtr HWnd_;              // New for 5.0
        protected char* PortName_;           // Required
        protected char* Application_;        // you can use %env vars% or %1
        protected char* DirectToFile_;       // 0 or 1
        protected char* DragDrop_;           // 0 or 1
        protected char* Extension_;          // Up to 3 characters
        protected char* FileHandling_;       // FH_ Constants
        protected char* Forwarding_;         // Port List, comma separated
        protected char* LogName_;            // Filename only
        protected char* LogOptions_;         // LO_ Constants
        protected char* NameOptions_;        // AF_ Constants
        protected char* Port_;               // Output path
        protected char* ShowWindow_;         // 0 Normal_; 1 Minimized_; 2 Maximized
        protected char* UserDir_;            // 0 or 1  NO!- UD_ constants
        protected char* Spooling_;           // Printer List, comma separated
        protected char* UserName_;           // NT Only_; Not returned by Get...
        protected char* Password_;           // NT Only_; Not returned by Get...
        protected char* Domain_;             // NT Only
        protected char* AppWaitOptions_;     // AW Constants
        protected char* AppWaitTimeout_;     // In milliseconds
        protected char* AppWaitDelete_;      // 0 or 1

        protected char* TCPPort_;            // The listening port for TCP/IP Printing
        protected char* TCPBindings_;        // A comma sparated list of bindings
        protected char* TCPOptions_;
        protected char* Enabled_;
        protected char* SnmpEnabled_;
        protected char* SnmpManufacturer_;
        protected char* SnmpHardwareId_;
        protected char* SnmpModel_;
        protected char* SnmpOptions_;
        protected char* DebugOptions_;
        protected char* TimeOptions_;
        protected char* Modified_;
        protected char* ModifiedBy_;
        protected char* ModifiedPreviously_;
        protected char* ModifiedPreviouslyBy_;

        public const int NumCharFields = 35; // increase if needed--must = numof(byte*) above
        private const int Padding = sizeof(UInt32);
        public int LastWin32Error = 0;

        // should be a MAX_PATH, but since C# doesn't do defines, we're stuck with 
        // manifest constants in our code
        public const int StringSize = 260;
        // counters to actually store the byte offsets where the port capture expects to 
        // get our data
        protected char* iP;
        protected IntPtr p;

        public AVPMonitor()
        {
            if (IntPtr.Size == 8)
            {
                // Field alignment changes depending on the CPU platform
                Size_ = (UInt32)(sizeof(UInt32) + sizeof(IntPtr) + Padding + (NumCharFields * sizeof(IntPtr)));
            } else
            {
                Size_ = (UInt32)(sizeof(UInt32) + sizeof(IntPtr) + (NumCharFields * sizeof(IntPtr)));
            }
            HWnd_ = IntPtr.Zero; // Version 6.0 does not use this
            p = Marshal.AllocHGlobal(NumCharFields * (StringSize * 2));  // For Unicode
            iP = (char*)p.ToPointer(); // p.ToInt32();
            PortName_ = incPtr();
            Application_ = incPtr();
            DirectToFile_ = incPtr();
            DragDrop_ = incPtr();
            Extension_ = incPtr();
            FileHandling_ = incPtr();
            Forwarding_ = incPtr();
            LogName_ = incPtr();
            LogOptions_ = incPtr();
            NameOptions_ = incPtr();
            Port_ = incPtr();
            ShowWindow_ = incPtr();
            UserDir_ = incPtr();
            Spooling_ = incPtr();
            UserName_ = incPtr();
            Password_ = incPtr();
            Domain_ = incPtr();
            AppWaitOptions_ = incPtr();
            AppWaitTimeout_ = incPtr();
            AppWaitDelete_ = incPtr();
            TCPPort_ = incPtr();
            TCPBindings_ = incPtr();
            TCPOptions_ = incPtr();
            Enabled_ = incPtr();
            SnmpEnabled_ = incPtr();
            SnmpManufacturer_ = incPtr();
            SnmpHardwareId_ = incPtr();
            SnmpModel_ = incPtr();
            SnmpOptions_ = incPtr();
            DebugOptions_ = incPtr();
            TimeOptions_ = incPtr();
            Modified_ = incPtr();
            ModifiedBy_ = incPtr();
            ModifiedPreviously_ = incPtr();
            ModifiedPreviouslyBy_ = incPtr();
        }

        ~AVPMonitor()
        {
            if (p != IntPtr.Zero)
                Marshal.FreeHGlobal(p);
        }

        /// <summary>
        /// Sets up the data locations for fields, initializes to all byte(0)
        /// </summary>
        /// <returns>pointer to the location of the field</returns>
        protected char* incPtr()
        {
            IntPtr Q = new IntPtr(iP);
            char* b = (char*)(Q.ToPointer());
            char* c = b;
            char* ec = c + StringSize;
            while (c != ec)
                *c++ = (char)0;
            iP += StringSize;
            return b;
        }

        /// <summary>
        /// intrinsically unsafe--copy a byte * from a string
        /// </summary>
        /// <param name="p"></param>
        /// <param name="s"></param>
        public static void StringToByte(char* p, string s)
        {
            char* c = p;
            char* ec = p + StringSize;
            while (c != ec)
                *c++ = (char)0;
            c = p;
            for (int i = 0; i < s.Length && i < StringSize; i++)
            {
                *c++ = (char)s[i];
            }
        }

        /// <summary>
        /// intrinsically unsafe--copy a byte * to a string
        /// </summary>
        /// <param name="p"></param>
        /// <param name="s"></param>
        public static string ByteToString(char* b)
        {
            return new string((char*)b);
        }

        //accessors
        public string PortName
        {
            get
            {
                return ByteToString(PortName_);
            }
            set
            {
                StringToByte(PortName_, value);
            }
        }

        public string Application
        {
            get
            {
                return ByteToString(Application_);
            }
            set
            {
                StringToByte(Application_, value);
            }
        }

        public string Extension
        {
            get
            {
                return ByteToString(Extension_);
            }
            set
            {
                StringToByte(Extension_, value);
            }
        }

        public string ForwardingPorts
        {
            get
            {
                return ByteToString(Forwarding_);
            }
            set
            {
                StringToByte(Forwarding_, value);
            }
        }

        public string LogName
        {
            get
            {
                return ByteToString(LogName_);
            }
            set
            {
                StringToByte(LogName_, value);
            }
        }

        public string OutputPath
        {
            get
            {
                return ByteToString(Port_);
            }
            set
            {
                StringToByte(Port_, value);
            }
        }

        public string SpoolingList
        {
            get
            {
                return ByteToString(Spooling_);
            }
            set
            {
                StringToByte(Spooling_, value);
            }
        }

        public string UserName
        {
            get
            {
                return ByteToString(UserName_);
            }
            set
            {
                StringToByte(UserName_, value);
            }
        }

        public string Password
        {
            get
            {
                return ByteToString(Password_);
            }
            set
            {
                StringToByte(Password_, value);
            }
        }

        public string Domain
        {
            get
            {
                return ByteToString(Domain_);
            }
            set
            {
                StringToByte(Domain_, value);
            }
        }

        public const char ByteOne = '1';
        public const char ByteZero = '0';
        public bool DirectToFile
        {
            get
            {
                return (*DirectToFile_ == ByteOne);
            }
            set
            {
                *DirectToFile_ = value ? ByteOne : ByteZero;
            }
        }

        public bool DragDrop
        {
            get
            {
                return (*DragDrop_ == ByteOne);
            }
            set
            {
                *DragDrop_ = value ? ByteOne : ByteZero;
            }
        }

        public eUserDirOptions UserDir
        {
            get
            {
                return (eUserDirOptions)ByteToInt(UserDir_, 0);
            }
            set
            {
                IntToByte(UserDir_, (int)value);
            }
        }

        public bool AppWaitDelete
        {
            get
            {
                return (*AppWaitDelete_ == ByteOne);
            }
            set
            {
                *AppWaitDelete_ = value ? ByteOne : ByteZero;
            }
        }

        /// <summary>
        /// Post Processing Application wait options must be ON.
        /// Timeout wait in milliseconds
        /// </summary>
        public int AppWaitTimeout
        {
            get
            {
                return ByteToInt(AppWaitTimeout_, 0);
            }
            set
            {
                IntToByte(AppWaitTimeout_, value);
            }
        }

        /// <summary>
        /// Post Processing Application wait options
        /// </summary>
        public eAppWaitOptions AppWaitOptions
        {
            get
            {
                return (eAppWaitOptions)ByteToInt(AppWaitOptions_, 0);
            }
            set
            {
                IntToByte(AppWaitOptions_, (int)value);
            }
        }

        /// <summary>
        /// File handling options from FH_
        /// </summary>
        public eFileHandlingOptions FileHandlingOptions
        {
            get
            {
                return (eFileHandlingOptions)ByteToInt(FileHandling_, 0);
            }
            set
            {
                IntToByte(FileHandling_, (int)value);
            }
        }

        /// <summary>
        /// Log File options
        /// </summary>
        public eLogOptions LogOptions
        {
            get
            {
                return (eLogOptions)ByteToInt(LogOptions_, 0);
            }
            set
            {
                IntToByte(LogOptions_, (int)value);
            }
        }

        /// <summary>
        /// File Naming conventions for the output file 
        /// </summary>
        public eNameOptions NameOptions
        {
            get
            {
                return (eNameOptions)ByteToInt(NameOptions_, 0);
            }
            set
            {
                IntToByte(NameOptions_, (int)value);
            }
        }

        public eHandlerPgmOptions ShowWindow
        {
            get
            {
                return (eHandlerPgmOptions)ByteToInt(ShowWindow_, 0);
            }
            set
            {
                IntToByte(ShowWindow_, (int)value);
            }
        }

        public int TCPPort
        {
            get
            {
                return ByteToInt(TCPPort_, 0);
            }
            set
            {
                IntToByte(TCPPort_, value);
            }
        }
        public string TCPBindings
        {
            get
            {
                return ByteToString(TCPBindings_);
            }
            set
            {
                StringToByte(TCPBindings_, value);
            }
        }
        public eTcpOptions TCPOptions
        {
            get
            {
                return (eTcpOptions)ByteToInt(TCPOptions_, 0);
            }
            set
            {
                IntToByte(TCPOptions_, (int)value);
            }
        }
        public bool Enabled
        {
            get
            {
                return (*Enabled_ == ByteOne);
            }
            set
            {
                *Enabled_ = value ? ByteOne : ByteZero;
            }
        }
        public bool SnmpEnabled
        {
            get
            {
                return (*SnmpEnabled_ == ByteOne);
            }
            set
            {
                *SnmpEnabled_ = value ? ByteOne : ByteZero;
            }
        }
        public string SnmpManufacturer
        {
            get
            {
                return ByteToString(SnmpManufacturer_);
            }
            set
            {
                StringToByte(SnmpManufacturer_, value);
            }
        }
        public string SnmpHardwareId
        {
            get
            {
                return ByteToString(SnmpHardwareId_);
            }
            set
            {
                StringToByte(SnmpHardwareId_, value);
            }
        }
        public string SnmpModel
        {
            get
            {
                return ByteToString(SnmpModel_);
            }
            set
            {
                StringToByte(SnmpModel_, value);
            }
        }

        public eSnmpOptions SnmpOptions
        {
            get
            {
                return (eSnmpOptions)ByteToInt(SnmpOptions_, 0);
            }
            set
            {
                IntToByte(SnmpOptions_, (int)value);
            }

        }

        public eDebugOptions DebugOptions
        {
            get
            {
                return (eDebugOptions)ByteToInt(DebugOptions_, 0);
            }
            set
            {
                IntToByte(DebugOptions_, (int)value);
            }
        }

        public eTimeOptions TimeOptions
        {
            get
            {
                return (eTimeOptions)ByteToInt(TimeOptions_, 0);
            }
            set
            {
                IntToByte(TimeOptions_, (int)value);
            }
        }

        public DateTime Modified
        {
            get
            {
                string s = ByteToString(Modified_);
                if (s.Length > 0)
                {
                    return DateTime.Parse(s);
                }
                else
                {
                    return InvalidDateTime;
                }

            }
            set
            {
                StringToByte(Modified_, value.ToString("g",DateTimeFormatInfo.InvariantInfo));

            }
        }

        public string ModifiedBy
        {
            get
            {
                return ByteToString(ModifiedBy_);
            }
            set
            {
                StringToByte(ModifiedBy_, value);
            }
        }

        public DateTime ModifiedPreviously
        {
            get
            {
                string s = ByteToString(ModifiedPreviously_);
                if (s.Length > 0)
                {
                    return DateTime.Parse(s);
                }
                else
                {
                    return InvalidDateTime;
                }

            }
            set
            {
                StringToByte(ModifiedPreviously_, value.ToString("g", DateTimeFormatInfo.InvariantInfo));

            }
        }

        public string ModifiedPreviouslyBy
        {
            get
            {
                return ByteToString(ModifiedPreviouslyBy_);
            }
            set
            {
                StringToByte(Domain_, value);
            }
        }

        public DateTime InvalidDateTime
        {
            get
            {
                return new DateTime(0);
            }
        }



        protected static void IntToByte(char* p, int val)
        {
            StringToByte(p, val.ToString());
        }

        protected static int ByteToInt(char* p, int DefaultValue)
        {
            int ret = DefaultValue;
            try
            {
                string s = ByteToString(p);
                if (!String.IsNullOrEmpty(s))
                {
                    ret = Convert.ToInt32(s);
                }
            }
            catch { }
            return ret;
        }

        public enum eFileHandlingOptions : int
        {
            FH_AutoFile = 0,
            FH_NoFile = 1,
            FH_LogFile = 2,
        }

        public enum eNameOptions : int
        {
            AF_Title = 0,
            AF_DateTime = 1,
            AF_TitleDateTime = 2,
            AF_DateTimeTitle = 3,
            AF_UserNameDateTime = 4,
            AF_DateTimeUserName = 5,
            AF_ComputerNameDateTime = 6,
            AF_DateTimeComputerName = 7,
            AF_PortDateTime = 8,
            AF_DateTimePort = 9,
            AF_TitleUserNameDateTime = 10,
            AF_TitleDateTimeUserName = 11,
            AF_UserNameTitleDateTime = 12,
            AF_UserNameDateTimeTitle = 13,
            AF_DateTimeUserNameTitle = 14,
            AF_DateTimeTitleUserName = 15,
            AF_TitleComputerNameDateTime = 16,
            AF_TitleDateTimeComputerName = 17,
            AF_ComputerNameTitleDateTime = 18,
            AF_ComputerNameDateTimeTitle = 19,
            AF_DateTimeComputerNameTitle = 20,
            AF_DateTimeTitleComputerName = 21,
        }

        public enum eLogOptions : int
        {
            LO_Day = 0,
            LO_Month = 1,
            LO_DayN = 2,
            LO_MonthN = 3,
            LO_YearN = 4,
            LO_File = 5,
        }

        public enum eHandlerPgmOptions : int
        {
            SW_Normal = 0,
            SW_Minimized = 1,
            SW_Maximized = 2,
        }

        public enum eAppWaitOptions : int
        {
            AW_None = 0,
            AW_Infinite = 1,
            AW_UseTimeout = 2,
        }

        public enum eUserDirOptions : int
        {
            UD_Base = 0,
            UD_User = 1,
            UD_Computer = 2,
            UD_Printer = 3,
            UD_Driver = 4
        }

        public enum eTcpOptions : int
        {
            TO_Query = 0,
            TO_Host = 1,
            TO_None = 2
        }

        public enum eSnmpOptions : int
        {
            SO_Auto = 0,
            SO_Dll = 1,
            SO_BuiltIn = 2
        }

        public enum eDebugOptions : int
        {
            DO_Disabled = 0,
            DO_Always = 1,
            DO_Error = 2
        }


        public enum eTimeOptions : int
        {
            TM_Utc = 0,
            TM_Local = 1
        }


        public bool SetVirtualPort()
        {

            uint r = SetVirtualPort(this);
            LastWin32Error = Marshal.GetLastWin32Error();
            return r != 0;

        }

        public bool GetVirtualPort(string PortName)
        {
            this.PortName = PortName;
            uint r = GetVirtualPort(this, StringSize);
            LastWin32Error = Marshal.GetLastWin32Error();
            return r != 0;
        }

        [DllImport("AVPDlg.dll", CallingConvention = CallingConvention.Winapi, CharSet = CharSet.Unicode)]
        static extern UInt32 SetVirtualPort(AVPMonitor VirtualPort);

        /// Pass a pointer to a TVirtualPort Structure and the
        /// Per Member buffer size. ie; Each string can hold 1024 bytes
        [DllImport("AVPDlg.dll", CallingConvention = CallingConvention.Winapi, CharSet = CharSet.Unicode)]
        static extern UInt32 GetVirtualPort(AVPMonitor VirtualPort, UInt32 BufferSize);

    }
}
